<?php
/**
 * Plugin Name: GrL'Ca Divi Shop View Switcher
 * Plugin URI: https://grlca.com
 * Description: Adds grid/list view switcher for Divi WooCommerce shops
 * Tested up to: WP 6.8
 * Version: 1.0.4
 * Author: GrL'Ca
 * Author URI: https://grlca.com/
 * Tags: divi, divi shop, divi view, divi product module
 * License: GPL v2 or later
 * Text Domain: grlca-divi-shop-view-switcher
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Include the GrL'Ca Dashboard
require_once plugin_dir_path(__FILE__) . 'grlca-dashboard.php';

class DiviShopViewSwitcher {
    
    private $options;
    
    public function __construct() {
        $this->options = get_option('grlca_divi_shop_switcher_options');
        
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_head', array($this, 'add_initial_styles'));
        add_action('wp_footer', array($this, 'add_switcher_script'));
        
        // Admin
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'admin_init'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
    }
    
    public function admin_enqueue_scripts($hook) {
        if ('grlca_page_grlca-divi-shop-switcher' !== $hook) {
            return;
        }
        
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }
    
    public function add_admin_menu() {
        // Add as submenu under GrL'Ca parent menu
        add_submenu_page(
            'grlca-dashboard',
            'Shop View Switcher Settings',
            'Shop Switcher',
            'manage_options',
            'grlca-divi-shop-switcher',
            array($this, 'admin_page')
        );
    }
    
    public function admin_init() {
        register_setting('grlca_divi_shop_switcher', 'grlca_divi_shop_switcher_options');
        
        add_settings_section(
            'grlca_divi_shop_switcher_section',
            'Visual Controls',
            array($this, 'section_callback'),
            'grlca_divi_shop_switcher'
        );
        
        // View Switcher Settings
        add_settings_field(
            'switcher_bg_color',
            'Switcher Active Background',
            array($this, 'color_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_bg_color', 'default' => '#2ea3f2')
        );
        
        add_settings_field(
            'switcher_text_color',
            'Switcher Active Text Color',
            array($this, 'color_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_text_color', 'default' => '#ffffff')
        );
        
        add_settings_field(
            'switcher_inactive_bg',
            'Switcher Inactive Background',
            array($this, 'color_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_inactive_bg', 'default' => '#f8f9fa')
        );
        
        add_settings_field(
            'switcher_inactive_text',
            'Switcher Inactive Text Color',
            array($this, 'color_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_inactive_text', 'default' => '#666666')
        );
        
        add_settings_field(
            'switcher_font_size',
            'Switcher Font Size (px)',
            array($this, 'number_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_font_size', 'default' => '14', 'min' => '8', 'max' => '24')
        );
        
        add_settings_field(
            'switcher_padding',
            'Switcher Padding (px)',
            array($this, 'number_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_padding', 'default' => '10', 'min' => '5', 'max' => '30')
        );
        
        add_settings_field(
            'switcher_border_radius',
            'Switcher Border Radius (px)',
            array($this, 'number_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'switcher_border_radius', 'default' => '25', 'min' => '0', 'max' => '50')
        );
        
        // View Name Settings
        add_settings_field(
            'grid_view_name',
            'Grid View Name',
            array($this, 'text_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'grid_view_name', 'default' => 'Grid')
        );
        
        add_settings_field(
            'list_view_name',
            'List View Name',
            array($this, 'text_field_render'),
            'grlca_divi_shop_switcher',
            'grlca_divi_shop_switcher_section',
            array('field' => 'list_view_name', 'default' => 'List')
        );
    }
    
    public function color_field_render($args) {
        $field = $args['field'];
        $default = $args['default'];
        $value = isset($this->options[$field]) ? $this->options[$field] : $default;
        ?>
        <input type="text" name="grlca_divi_shop_switcher_options[<?php echo $field; ?>]" value="<?php echo $value; ?>" class="color-picker" />
        <?php
    }
    
    public function number_field_render($args) {
        $field = $args['field'];
        $default = $args['default'];
        $min = $args['min'];
        $max = $args['max'];
        $value = isset($this->options[$field]) ? $this->options[$field] : $default;
        ?>
        <input type="number" name="grlca_divi_shop_switcher_options[<?php echo $field; ?>]" value="<?php echo $value; ?>" min="<?php echo $min; ?>" max="<?php echo $max; ?>" style="width: 80px;" /> px
        <?php
    }
    
    public function text_field_render($args) {
        $field = $args['field'];
        $default = $args['default'];
        $value = isset($this->options[$field]) ? $this->options[$field] : $default;
        ?>
        <input type="text" name="grlca_divi_shop_switcher_options[<?php echo $field; ?>]" value="<?php echo $value; ?>" style="width: 200px;" />
        <?php
    }
    
    public function section_callback() {
        echo 'Customize the appearance of the View Switcher. Changes will be applied immediately after saving.';
    }
    
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>Shop View Switcher Settings</h1>
            
            <div style="background: #fff; padding: 20px; margin-top: 20px; border-radius: 5px;">
                <form action="options.php" method="post">
                    <?php
                    settings_fields('grlca_divi_shop_switcher');
                    do_settings_sections('grlca_divi_shop_switcher');
                    submit_button();
                    ?>
                </form>
            </div>
            
            <div style="background: #fff; padding: 20px; margin-top: 20px; border-radius: 5px;">
                <h3>Preview</h3>
                <p>This is a rough preview of how your view switcher will look:</p>
                
                <div style="display: flex; gap: 20px; flex-wrap: wrap; margin: 20px 0;">
                    <div>
                        <h4>View Switcher</h4>
                        <div style="display: flex; gap: 10px;">
                            <button style="
                                background: <?php echo isset($this->options['switcher_bg_color']) ? $this->options['switcher_bg_color'] : '#2ea3f2'; ?>;
                                color: <?php echo isset($this->options['switcher_text_color']) ? $this->options['switcher_text_color'] : '#ffffff'; ?>;
                                font-size: <?php echo isset($this->options['switcher_font_size']) ? $this->options['switcher_font_size'] : '14'; ?>px;
                                padding: <?php echo isset($this->options['switcher_padding']) ? $this->options['switcher_padding'] : '10'; ?>px 16px;
                                border-radius: <?php echo isset($this->options['switcher_border_radius']) ? $this->options['switcher_border_radius'] : '25'; ?>px;
                                border: none;
                                cursor: pointer;
                            ">▦ <?php echo isset($this->options['grid_view_name']) ? $this->options['grid_view_name'] : 'Grid'; ?></button>
                            
                            <button style="
                                background: <?php echo isset($this->options['switcher_inactive_bg']) ? $this->options['switcher_inactive_bg'] : '#f8f9fa'; ?>;
                                color: <?php echo isset($this->options['switcher_inactive_text']) ? $this->options['switcher_inactive_text'] : '#666666'; ?>;
                                font-size: <?php echo isset($this->options['switcher_font_size']) ? $this->options['switcher_font_size'] : '14'; ?>px;
                                padding: <?php echo isset($this->options['switcher_padding']) ? $this->options['switcher_padding'] : '10'; ?>px 16px;
                                border-radius: <?php echo isset($this->options['switcher_border_radius']) ? $this->options['switcher_border_radius'] : '25'; ?>px;
                                border: none;
                                cursor: pointer;
                            ">☰ <?php echo isset($this->options['list_view_name']) ? $this->options['list_view_name'] : 'List'; ?></button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('.color-picker').wpColorPicker();
        });
        </script>
        
        <style>
        .wp-picker-container .wp-color-result.button {
            height: 30px;
        }
        </style>
        <?php
    }
    
    public function enqueue_scripts() {
        if (!(is_shop() || is_product_category())) {
            return;
        }
        
        if (function_exists('is_product') && function_exists('wc_get_script_data')) {
            wp_enqueue_script('wc-add-to-cart');
        }
    }
    
    public function add_initial_styles() {
        if (!(is_shop() || is_product_category())) {
            return;
        }
        ?>
        <style type="text/css">
        /* Initial hidden state to prevent flash */
        .et_pb_shop .products {
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        .et_pb_shop .products.initialized {
            opacity: 1;
        }
        
        /* View Switcher - Dynamic Styles */
        .shop-view-switcher {
            text-align: right;
            margin-bottom: 20px;
            padding: 0 10px;
        }
        .shop-view-switcher .view-btn {
            background: <?php echo isset($this->options['switcher_inactive_bg']) ? $this->options['switcher_inactive_bg'] : '#f8f9fa'; ?>;
            border: none;
            padding: <?php echo isset($this->options['switcher_padding']) ? $this->options['switcher_padding'] : '10'; ?>px 16px;
            border-radius: <?php echo isset($this->options['switcher_border_radius']) ? $this->options['switcher_border_radius'] : '25'; ?>px;
            margin-left: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            font-size: <?php echo isset($this->options['switcher_font_size']) ? $this->options['switcher_font_size'] : '14'; ?>px;
            font-weight: 600;
            color: <?php echo isset($this->options['switcher_inactive_text']) ? $this->options['switcher_inactive_text'] : '#666666'; ?>;
            box-shadow: 0 1px 4px rgba(0, 0, 0, 0.1);
        }
        .shop-view-switcher .view-btn.active {
            background: <?php echo isset($this->options['switcher_bg_color']) ? $this->options['switcher_bg_color'] : '#2ea3f2'; ?>;
            color: <?php echo isset($this->options['switcher_text_color']) ? $this->options['switcher_text_color'] : '#ffffff'; ?>;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
        }
        .shop-view-switcher .view-btn:hover {
            background: <?php echo isset($this->options['switcher_bg_color']) ? $this->options['switcher_bg_color'] : '#2ea3f2'; ?>;
            color: <?php echo isset($this->options['switcher_text_color']) ? $this->options['switcher_text_color'] : '#ffffff'; ?>;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
        }
        .shop-view-switcher .view-icon {
            font-size: 16px;
        }
        .shop-view-switcher .view-text {
            font-size: <?php echo isset($this->options['switcher_font_size']) ? $this->options['switcher_font_size'] : '14'; ?>px;
        }

        /* NUCLEAR OPTION - COMPLETELY OVERRIDE DIVI GRID SYSTEM FOR LIST VIEW */
        body.shop-list .et_pb_shop .products {
            display: block !important;
            width: 100% !important;
            max-width: 100% !important;
            margin: 0 !important;
            padding: 0 !important;
            opacity: 1 !important;
            visibility: visible !important;
        }

        body.shop-list .et_pb_shop .product {
            display: flex !important;
            flex-direction: column !important;
            width: 100% !important;
            max-width: 100% !important;
            float: none !important;
            clear: both !important;
            margin: 0 !important;
            padding: 15px 0 !important;
            border-bottom: 1px solid #eaeaea;
            box-shadow: none !important;
            background: transparent !important;
            position: relative !important;
            left: auto !important;
            right: auto !important;
            top: auto !important;
            bottom: auto !important;
        }

        body.shop-list .et_pb_shop .product:last-child {
            border-bottom: none;
        }

        /* Hide images in list view */
        body.shop-list .et_pb_shop .product img {
            display: none !important;
        }

        /* Title - Full width, multiple lines */
        body.shop-list .et_pb_shop .product .woocommerce-loop-product__title {
            display: block !important;
            width: 100% !important;
            max-width: 100% !important;
            font-size: 16px !important;
            font-weight: 600;
            margin: 0 0 8px 0 !important;
            padding: 0 15px !important;
            line-height: 1.4 !important;
            text-align: left;
            white-space: normal !important;
            overflow: visible !important;
            text-overflow: unset !important;
            visibility: visible !important;
            opacity: 1 !important;
        }

        /* Actions container - Full width row */
        body.shop-list .et_pb_shop .product .product-actions {
            display: flex !important;
            width: 100% !important;
            max-width: 100% !important;
            justify-content: space-between !important;
            align-items: center !important;
            padding: 0 15px !important;
            gap: 15px;
        }

        /* Price */
        body.shop-list .et_pb_shop .product .price {
            font-weight: 600;
            color: #222;
            font-size: 16px !important;
            margin: 0 !important;
            text-align: left !important;
            white-space: nowrap !important;
            visibility: visible !important;
            opacity: 1 !important;
            flex-shrink: 0;
        }

        /* Add to Cart Button */
        body.shop-list .et_pb_shop .product a.button,
        body.shop-list .et_pb_shop .product .button {
            white-space: nowrap !important;
            margin: 0 !important;
            padding: 8px 16px !important;
            font-size: 14px !important;
            min-width: 120px !important;
            text-align: center;
            visibility: visible !important;
            opacity: 1 !important;
            flex-shrink: 0;
        }

        /* GRID VIEW - DEFAULT DIVI STYLING */
        body.shop-grid .et_pb_shop .products {
            /* Let Divi handle all grid styling */
        }

        body.shop-grid .et_pb_shop .product {
            /* Let Divi handle all product styling */
        }

        /* Hide descriptions in both views */
        body.shop-grid .et_pb_shop .product .woocommerce-product-details__short-description,
        body.shop-list .et_pb_shop .product .woocommerce-product-details__short-description {
            display: none !important;
        }

        /* MOBILE STYLES - 100VW FIX FOR ALL SCREENS UNDER 980PX */
        @media (max-width: 980px) {
            .shop-view-switcher .view-btn .view-text {
                display: none;
            }
            
            /* Break out of container for all mobile screens */
            body.shop-list .et_pb_shop {
                width: 100vw !important;
                max-width: 100vw !important;
                margin-left: calc(-50vw + 50%) !important;
                margin-right: calc(-50vw + 50%) !important;
                padding-left: 0 !important;
                padding-right: 0 !important;
            }

            body.shop-list .et_pb_shop .products {
                width: 100vw !important;
                max-width: 100vw !important;
                margin-left: 0 !important;
                margin-right: 0 !important;
            }

            body.shop-list .et_pb_shop .product {
                width: 100vw !important;
                max-width: 100vw !important;
                padding: 12px 5% !important;
                margin: 0 !important;
            }

            body.shop-list .et_pb_shop .product .woocommerce-loop-product__title {
                font-size: 15px !important;
                padding: 0 !important;
                margin-bottom: 6px !important;
                white-space: normal !important;
                overflow: visible !important;
                text-overflow: unset !important;
                line-height: 1.4 !important;
                word-wrap: break-word !important;
                word-break: break-word !important;
            }

            body.shop-list .et_pb_shop .product .product-actions {
                padding: 0 !important;
                gap: 12px;
            }

            body.shop-list .et_pb_shop .product .price {
                font-size: 15px !important;
            }

            body.shop-list .et_pb_shop .product a.button,
            body.shop-list .et_pb_shop .product .button {
                font-size: 13px !important;
                padding: 7px 14px !important;
                min-width: 110px !important;
            }
        }

        @media (max-width: 768px) {
            body.shop-list .et_pb_shop .product {
                padding: 10px 5% !important;
            }

            body.shop-list .et_pb_shop .product .woocommerce-loop-product__title {
                font-size: 14px !important;
            }

            body.shop-list .et_pb_shop .product .product-actions {
                gap: 10px;
            }

            body.shop-list .et_pb_shop .product .price {
                font-size: 14px !important;
            }

            body.shop-list .et_pb_shop .product a.button,
            body.shop-list .et_pb_shop .product .button {
                font-size: 12px !important;
                padding: 6px 12px !important;
                min-width: 100px !important;
            }
        }

        @media (max-width: 480px) {
            body.shop-list .et_pb_shop .product {
                padding: 8px 5% !important;
            }

            body.shop-list .et_pb_shop .product .woocommerce-loop-product__title {
                font-size: 13px !important;
                line-height: 1.3 !important;
            }

            body.shop-list .et_pb_shop .product .product-actions {
                gap: 8px;
                flex-wrap: wrap;
                justify-content: space-between !important;
            }

            body.shop-list .et_pb_shop .product .price {
                font-size: 13px !important;
                margin-right: 0 !important;
            }

            body.shop-list .et_pb_shop .product a.button,
            body.shop-list .et_pb_shop .product .button {
                font-size: 11px !important;
                padding: 5px 10px !important;
                min-width: 90px !important;
            }
        }

        /* EXTREME OVERRIDE - Target any potential Divi grid classes */
        body.shop-list .et_pb_shop .products .et_pb_module,
        body.shop-list .et_pb_shop .products .et_pb_column,
        body.shop-list .et_pb_shop .products .et_pb_row,
        body.shop-list .et_pb_shop .products .woocommerce.columns-*,
        body.shop-list .et_pb_shop .products .columns-* {
            width: 100% !important;
            max-width: 100% !important;
            float: none !important;
            margin: 0 !important;
            padding: 0 !important;
        }

        body.shop-list .et_pb_shop .product.et_pb_grid_item,
        body.shop-list .et_pb_shop .product.et_pb_portfolio_item {
            width: 100% !important;
            max-width: 100% !important;
            margin: 0 !important;
            padding: 15px 0 !important;
        }
        </style>
        <?php
    }
    
    public function add_switcher_script() {
        if (!(is_shop() || is_product_category())) {
            return;
        }
        
        // Get the custom view names or use defaults
        $grid_view_name = isset($this->options['grid_view_name']) ? $this->options['grid_view_name'] : 'Grid';
        $list_view_name = isset($this->options['list_view_name']) ? $this->options['list_view_name'] : 'List';
        ?>
        <script type="text/javascript">
        (function($) {
            'use strict';
            
            // Configuration
            const shopViewConfig = {
                localStorageKey: 'divi_shop_view_preference',
                defaultView: 'grid'
            };
            
            // Text translations - NOW CUSTOMIZABLE
            const textTranslations = {
                gridView: '<?php echo esc_js($grid_view_name); ?>',
                listView: '<?php echo esc_js($list_view_name); ?>'
            };
            
            let isInitialized = false;
            let originalProductHTML = new Map(); // Store original HTML for each product
            
            // === Apply View Immediately ===
            function applyViewImmediately() {
                const savedView = getSavedView();
                
                // Remove any existing view classes
                document.body.classList.remove('shop-grid', 'shop-list');
                // Add the saved view class immediately
                document.body.classList.add(`shop-${savedView}`);
                
                // Show the content immediately without waiting for DOM ready
                initializeProductDisplay();
            }
            
            // === Initialize Product Display ===
            function initializeProductDisplay() {
                const products = document.querySelector('.et_pb_shop .products');
                if (products) {
                    products.classList.add('initialized');
                }
            }
            
            // === Store original product HTML ===
            function storeOriginalProductHTML() {
                const products = document.querySelectorAll('.et_pb_shop .product');
                products.forEach((product, index) => {
                    if (!originalProductHTML.has(index)) {
                        originalProductHTML.set(index, product.innerHTML);
                    }
                });
            }
            
            // === Helper: Setup List View Layout ===
            function setupListviewLayout() {
                const products = document.querySelectorAll('body.shop-list .et_pb_shop .product');
                
                // First, store original HTML if not already stored
                storeOriginalProductHTML();
                
                products.forEach((product, index) => {
                    // Skip if already processed
                    if (product.classList.contains('list-layout-setup')) return;
                    
                    const originalHTML = originalProductHTML.get(index);
                    const productClone = document.createElement('div');
                    productClone.innerHTML = originalHTML;
                    
                    // Get elements from original structure
                    const title = productClone.querySelector('.woocommerce-loop-product__title');
                    const price = productClone.querySelector('.price');
                    const addToCart = productClone.querySelector('a.button, .button');
                    
                    // Create new list view structure
                    const newHTML = `
                        ${title ? title.outerHTML : ''}
                        <div class="product-actions">
                            ${price ? price.outerHTML : ''}
                            ${addToCart ? addToCart.outerHTML : ''}
                        </div>
                    `;
                    
                    product.innerHTML = newHTML;
                    product.classList.add('list-layout-setup');
                });
            }
            
            // === Helper: Restore Grid View Layout ===
            function restoreGridViewLayout() {
                const products = document.querySelectorAll('body.shop-grid .et_pb_shop .product');
                
                products.forEach((product, index) => {
                    // Remove list view classes
                    product.classList.remove('list-layout-setup');
                    
                    // Restore original HTML
                    const originalHTML = originalProductHTML.get(index);
                    if (originalHTML) {
                        product.innerHTML = originalHTML;
                    }
                    
                    // Remove any inline styles
                    product.removeAttribute('style');
                    
                    // Ensure all elements are visible
                    const images = product.querySelectorAll('img');
                    const titles = product.querySelectorAll('.woocommerce-loop-product__title');
                    const prices = product.querySelectorAll('.price');
                    const buttons = product.querySelectorAll('a.button, .button');
                    
                    images.forEach(img => img.style.display = '');
                    titles.forEach(title => title.removeAttribute('style'));
                    prices.forEach(price => price.removeAttribute('style'));
                    buttons.forEach(button => button.removeAttribute('style'));
                });
            }
            
            // === View Switcher Logic ===
            function createViewSwitcher() {
                const shop = document.querySelector('.et_pb_shop');
                if (!shop || document.querySelector('.shop-view-switcher')) return;

                const savedView = getSavedView();

                const switcher = document.createElement('div');
                switcher.className = 'shop-view-switcher';
                switcher.innerHTML = `
                    <button class="view-btn ${savedView === 'grid' ? 'active' : ''}" data-view="grid">
                        <span class="view-icon">▦</span>
                        <span class="view-text">${textTranslations.gridView}</span>
                    </button>
                    <button class="view-btn ${savedView === 'list' ? 'active' : ''}" data-view="list">
                        <span class="view-icon">☰</span>
                        <span class="view-text">${textTranslations.listView}</span>
                    </button>
                `;
                
                shop.insertAdjacentElement('beforebegin', switcher);

                switcher.addEventListener('click', (e) => {
                    const btn = e.target.closest('.view-btn');
                    if (!btn) return;
                    const view = btn.dataset.view;
                    switchView(view, true);
                });
            }
            
            // === Get Saved View ===
            function getSavedView() {
                try {
                    const saved = localStorage.getItem(shopViewConfig.localStorageKey);
                    return (saved === 'grid' || saved === 'list') ? saved : shopViewConfig.defaultView;
                } catch (e) {
                    return shopViewConfig.defaultView;
                }
            }
            
            // === Switch View Function ===
            function switchView(view, save = true) {
                document.body.classList.remove('shop-grid', 'shop-list');
                document.body.classList.add(`shop-${view}`);
                
                document.querySelectorAll('.shop-view-switcher .view-btn').forEach(b => {
                    b.classList.toggle('active', b.dataset.view === view);
                });
                
                if (save) {
                    try {
                        localStorage.setItem(shopViewConfig.localStorageKey, view);
                    } catch (e) {
                        console.error('Could not save view preference:', e);
                    }
                }
                
                // Store original HTML before making any changes
                if (view === 'list') {
                    storeOriginalProductHTML();
                }
                
                // Force re-display of products immediately
                initializeProductDisplay();
                
                // Setup appropriate layout based on view
                if (view === 'list') {
                    setTimeout(setupListviewLayout, 10);
                } else {
                    setTimeout(restoreGridViewLayout, 10);
                }
            }
            
            // === Initialize Everything ===
            function initialize() {
                if (isInitialized) return;
                isInitialized = true;
                
                const savedView = getSavedView();
                document.body.classList.remove('shop-grid', 'shop-list');
                document.body.classList.add(`shop-${savedView}`);
                
                // Store original HTML on initial load
                storeOriginalProductHTML();
                
                createViewSwitcher();
                
                // Force display immediately
                initializeProductDisplay();
                
                // Setup appropriate layout based on current view
                if (savedView === 'list') {
                    setTimeout(setupListviewLayout, 50);
                } else {
                    setTimeout(restoreGridViewLayout, 50);
                }
            }
            
            // === Start Immediately ===
            applyViewImmediately();
            
            // Initialize when DOM is ready
            $(document).ready(function() {
                initialize();
                
                // Re-initialize after AJAX events (for pagination, etc.)
                $(document).ajaxComplete(function() {
                    setTimeout(() => {
                        isInitialized = false;
                        originalProductHTML.clear(); // Clear stored HTML
                        const currentView = getSavedView();
                        document.body.classList.remove('shop-grid', 'shop-list');
                        document.body.classList.add(`shop-${currentView}`);
                        initializeProductDisplay();
                        if (currentView === 'list') {
                            setupListviewLayout();
                        } else {
                            restoreGridViewLayout();
                        }
                    }, 100);
                });
            });
            
        })(jQuery);
        </script>
        <?php
    }
}

// Initialize the plugin
new DiviShopViewSwitcher();