(function($) {
    'use strict';
    
    let searchTimeout;
    
    $(document).ready(function() {
        // Initialize for both WooCommerce search form and our shortcode
        $('form[role="search"] input[name="s"]').each(function() {
            initSearchAutocomplete($(this));
        });
    });
    
    function initSearchAutocomplete($input) {
        const $suggestionsContainer = $input.closest('form').find('.grlca-woo-search-suggestions');
        const $form = $input.closest('form');
        
        $input.on('input', function() {
            const searchTerm = $(this).val().trim();
            
            // Clear previous timeout
            clearTimeout(searchTimeout);
            
            if (searchTerm.length < 2) {
                $suggestionsContainer.html('').hide();
                return;
            }
            
            // Set new timeout
            searchTimeout = setTimeout(function() {
                getSearchSuggestions(searchTerm, $suggestionsContainer);
            }, 300);
        });
        
        // Submit form on Enter key when no suggestion is selected
        $input.on('keydown', function(e) {
            if (e.key === 'Enter') {
                const $activeSuggestion = $suggestionsContainer.find('.grlca-woo-search-suggestion-item.active');
                if ($activeSuggestion.length === 0) {
                    // No active suggestion, submit the form
                    $form.submit();
                }
            }
            
            const $suggestions = $suggestionsContainer.find('.grlca-woo-search-suggestion-item');
            
            if (e.key === 'ArrowDown') {
                e.preventDefault();
                if ($suggestions.length > 0) {
                    const $first = $suggestions.first();
                    $suggestions.removeClass('active');
                    $first.addClass('active');
                    $first.focus();
                }
            }
        });
        
        // Hide suggestions when clicking outside
        $(document).on('click', function(e) {
            if (!$suggestionsContainer.is(e.target) && $suggestionsContainer.has(e.target).length === 0) {
                $suggestionsContainer.hide();
            }
        });
        
        $suggestionsContainer.on('keydown', '.grlca-woo-search-suggestion-item', function(e) {
            const $current = $(this);
            const $suggestions = $suggestionsContainer.find('.grlca-woo-search-suggestion-item');
            
            if (e.key === 'ArrowDown') {
                e.preventDefault();
                const $next = $current.next('.grlca-woo-search-suggestion-item');
                if ($next.length) {
                    $suggestions.removeClass('active');
                    $next.addClass('active').focus();
                }
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                const $prev = $current.prev('.grlca-woo-search-suggestion-item');
                if ($prev.length) {
                    $suggestions.removeClass('active');
                    $prev.addClass('active').focus();
                } else {
                    $suggestions.removeClass('active');
                    $input.focus();
                }
            } else if (e.key === 'Enter') {
                e.preventDefault();
                window.location.href = $current.find('a').attr('href');
            }
        });
    }
    
    function getSearchSuggestions(searchTerm, $container) {
        $.ajax({
            url: grlca_woo_search_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'grlca_woo_search_suggestions',
                search_term: searchTerm,
                nonce: grlca_woo_search_ajax.nonce
            },
            beforeSend: function() {
                $container.html('<div class="grlca-woo-search-loading">' + grlca_woo_search_ajax.loading_text + '</div>').show();
            },
            success: function(response) {
                if (response.success && response.data.length > 0) {
                    displaySuggestions(response.data, $container);
                } else {
                    $container.html('<div class="grlca-woo-search-no-results">' + grlca_woo_search_ajax.no_results_text + '</div>').show();
                }
            },
            error: function() {
                $container.html('<div class="grlca-woo-search-error">' + grlca_woo_search_ajax.error_text + '</div>').show();
            }
        });
    }
    
    function displaySuggestions(suggestions, $container) {
        let html = '<div class="grlca-woo-search-suggestions-list">';
        
        suggestions.forEach(function(suggestion) {
            if (suggestion.type === 'product') {
                html += `
                    <div class="grlca-woo-search-suggestion-item" tabindex="0">
                        <a href="${suggestion.url}">
                            <img src="${suggestion.image}" alt="${suggestion.title}" />
                            <div class="grlca-woo-search-suggestion-info">
                                <div class="grlca-woo-search-suggestion-title">${suggestion.title}</div>
                                <div class="grlca-woo-search-suggestion-price">${suggestion.price}</div>
                                ${suggestion.sku ? `<div class="grlca-woo-search-suggestion-sku">${grlca_woo_search_ajax.sku_text}: ${suggestion.sku}</div>` : ''}
                            </div>
                        </a>
                    </div>
                `;
            } else if (suggestion.type === 'category') {
                html += `
                    <div class="grlca-woo-search-suggestion-item" tabindex="0">
                        <a href="${suggestion.url}">
                            <div class="grlca-woo-search-suggestion-info">
                                <div class="grlca-woo-search-suggestion-title">${suggestion.title}</div>
                                <div class="grlca-woo-search-suggestion-category">${grlca_woo_search_ajax.category_text} (${suggestion.count} ${grlca_woo_search_ajax.products_text})</div>
                            </div>
                        </a>
                    </div>
                `;
            }
        });
        
        html += '</div>';
        $container.html(html).show();
    }
    
})(jQuery);