<?php
/**
 * Plugin Name: GrL'Ca Woo Search
 * Plugin URI: https://grlca.com
 * Description: Enhanced search functionality for WooCommerce with autocomplete suggestions
 * Version: 1.0.0
 * Author: GrL'Ca
 * Author URI: https://grlca.com
 * License: GPL v2 or later
 * Text Domain: grlca-woo-search
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Check if WooCommerce is active
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    add_action('admin_notices', function() {
        echo '<div class="error"><p>' . __('GrL\'Ca Woo Search requires WooCommerce to be installed and active.', 'grlca-woo-search') . '</p></div>';
    });
    return;
}

// Include dashboard functionality
require_once plugin_dir_path(__FILE__) . 'includes/grlca-dashboard.php';

class GrLca_Woo_Search {
    
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_grlca_woo_search_suggestions', array($this, 'ajax_search_suggestions'));
        add_action('wp_ajax_nopriv_grlca_woo_search_suggestions', array($this, 'ajax_search_suggestions'));
        add_shortcode('grlca_woo_search', array($this, 'search_shortcode'));
        
        // Hook into existing WooCommerce search form
        add_filter('get_product_search_form', array($this, 'enhance_search_form'));
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Initialize plugin
        add_action('init', array($this, 'init'));
    }
    
    public function init() {
        // Load plugin text domain
        load_plugin_textdomain('grlca-woo-search', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function add_admin_menu() {
        // Add submenu under GrL'Ca main menu
        add_submenu_page(
            'grlca-dashboard',
            __('Search Settings', 'grlca-woo-search'),
            __('Search', 'grlca-woo-search'),
            'manage_options',
            'grlca-woo-search-settings',
            array($this, 'display_settings_page')
        );
    }
    
    public function display_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'grlca-woo-search'));
        }
        ?>
        <div class="wrap">
            <h1><?php _e('GrL\'Ca Woo Search Settings', 'grlca-woo-search'); ?></h1>
            
            <div class="card">
                <h2><?php _e('Search Features', 'grlca-woo-search'); ?></h2>
                <p><?php _e('The GrL\'Ca Woo Search plugin enhances your WooCommerce store with:', 'grlca-woo-search'); ?></p>
                <ul>
                    <li><?php _e('Autocomplete search suggestions', 'grlca-woo-search'); ?></li>
                    <li><?php _e('Product images and prices in suggestions', 'grlca-woo-search'); ?></li>
                    <li><?php _e('Category suggestions', 'grlca-woo-search'); ?></li>
                    <li><?php _e('Keyboard navigation support', 'grlca-woo-search'); ?></li>
                    <li><?php _e('Mobile-responsive design', 'grlca-woo-search'); ?></li>
                </ul>
            </div>
            
            <div class="card">
                <h2><?php _e('How to Use', 'grlca-woo-search'); ?></h2>
                
                <h3><?php _e('Automatic Enhancement:', 'grlca-woo-search'); ?></h3>
                <p><?php _e('All existing WooCommerce search forms are automatically enhanced with autocomplete functionality.', 'grlca-woo-search'); ?></p>
                
                <h3><?php _e('Shortcode:', 'grlca-woo-search'); ?></h3>
                <p><?php _e('Use the shortcode to add search forms anywhere:', 'grlca-woo-search'); ?></p>
                <code>[grlca_woo_search]</code>
                
                <h4><?php _e('Custom Shortcode:', 'grlca-woo-search'); ?></h4>
                <code>[grlca_woo_search placeholder="Your custom placeholder" class="your-custom-class"]</code>
            </div>
            
            <div class="card">
                <h2><?php _e('Search Preview', 'grlca-woo-search'); ?></h2>
                <p><?php _e('Here\'s how the search form looks with autocomplete functionality:', 'grlca-woo-search'); ?></p>
                <div style="max-width: 400px; margin: 20px 0;">
                    <?php echo $this->search_shortcode(array()); ?>
                </div>
                <p><em><?php _e('Start typing to see autocomplete suggestions with product images and prices.', 'grlca-woo-search'); ?></em></p>
            </div>
            
            <div class="card">
                <h2><?php _e('Plugin Information', 'grlca-woo-search'); ?></h2>
                <p><strong><?php _e('Version:', 'grlca-woo-search'); ?></strong> 1.0.0</p>
                <p><strong><?php _e('Text Domain:', 'grlca-woo-search'); ?></strong> grlca-woo-search</p>
                <p><strong><?php _e('Languages:', 'grlca-woo-search'); ?></strong> English, Croatian</p>
                <p><strong><?php _e('Requirements:', 'grlca-woo-search'); ?></strong> WooCommerce</p>
            </div>
        </div>
        <?php
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('jquery');
        
        wp_enqueue_script(
            'grlca-woo-search',
            plugin_dir_url(__FILE__) . 'assets/js/grlca-woo-search.js',
            array('jquery'),
            '1.0.0',
            true
        );
        
        wp_enqueue_style(
            'grlca-woo-search',
            plugin_dir_url(__FILE__) . 'assets/css/grlca-woo-search.css',
            array(),
            '1.0.0'
        );
        
        // Localize script for AJAX with translations
        wp_localize_script('grlca-woo-search', 'grlca_woo_search_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grlca_woo_search_nonce'),
            'loading_text' => __('Loading...', 'grlca-woo-search'),
            'no_results_text' => __('No products found', 'grlca-woo-search'),
            'error_text' => __('Error loading suggestions', 'grlca-woo-search'),
            'category_text' => __('Category', 'grlca-woo-search'),
            'products_text' => __('products', 'grlca-woo-search'),
            'sku_text' => __('SKU', 'grlca-woo-search')
        ));
    }
    
    public function enhance_search_form($form) {
        // Remove the search button from WooCommerce search form
        $form = preg_replace('/<button type="submit"[^>]*>.*?<\/button>/', '', $form);
        
        // Add a wrapper and search suggestions container
        $form = str_replace('</form>', '<div class="grlca-woo-search-suggestions"></div></form>', $form);
        $form = '<div class="grlca-woo-search-wrapper">' . $form . '</div>';
        return $form;
    }
    
    public function ajax_search_suggestions() {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'grlca_woo_search_nonce')) {
            wp_die('Security check failed');
        }
        
        $search_term = sanitize_text_field($_POST['search_term']);
        
        if (empty($search_term)) {
            wp_die();
        }
        
        $suggestions = $this->get_search_suggestions($search_term);
        
        wp_send_json_success($suggestions);
    }
    
    private function get_search_suggestions($search_term) {
        $suggestions = array();
        
        // Search for products
        $products = new WP_Query(array(
            'post_type' => 'product',
            'posts_per_page' => 5,
            's' => $search_term,
            'post_status' => 'publish'
        ));
        
        foreach ($products->posts as $product_post) {
            $product = wc_get_product($product_post->ID);
            $image = wp_get_attachment_image_src(get_post_thumbnail_id($product_post->ID), 'thumbnail');
            
            $suggestions[] = array(
                'type' => 'product',
                'title' => $product_post->post_title,
                'url' => get_permalink($product_post->ID),
                'price' => $product->get_price_html(),
                'image' => $image ? $image[0] : wc_placeholder_img_src(),
                'sku' => $product->get_sku()
            );
        }
        
        // Search for product categories
        $categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'name__like' => $search_term,
            'number' => 3,
            'hide_empty' => true
        ));
        
        foreach ($categories as $category) {
            $suggestions[] = array(
                'type' => 'category',
                'title' => $category->name,
                'url' => get_term_link($category),
                'count' => $category->count
            );
        }
        
        return $suggestions;
    }
    
    public function search_shortcode($atts = array()) {
        $atts = shortcode_atts(array(
            'placeholder' => __('Search products...', 'grlca-woo-search'),
            'class' => ''
        ), $atts);
        
        ob_start();
        ?>
        <div class="grlca-woo-search-shortcode <?php echo esc_attr($atts['class']); ?>">
            <form role="search" method="get" action="<?php echo esc_url(home_url('/')); ?>">
                <input type="hidden" name="post_type" value="product" />
                <input type="search" 
                       class="search-field" 
                       placeholder="<?php echo esc_attr($atts['placeholder']); ?>" 
                       value="<?php echo get_search_query(); ?>" 
                       name="s" 
                       autocomplete="off" />
                <div class="grlca-woo-search-suggestions"></div>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }
}

new GrLca_Woo_Search();