<?php
/**
 * Main plugin class
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Prevent multiple class declarations
if ( class_exists( 'GrLca_Woo_Product_Slider_Manager' ) ) {
    return;
}

class GrLca_Woo_Product_Slider_Manager {

    private $post_type = 'woo_slider';

    public function __construct() {
        // Register post type early
        add_action( 'init', [ $this, 'register_slider_post_type' ] );
        
        // Load other functionality after init to avoid early translation issues
        add_action( 'init', [ $this, 'init_plugin' ] );
    }
    
    /**
     * Initialize plugin functionality after init hook
     * This prevents WooCommerce translation loading too early
     */
    public function init_plugin() {
        add_action( 'add_meta_boxes', [ $this, 'add_meta_boxes' ] );
        add_action( 'save_post', [ $this, 'save_slider_meta' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
        add_shortcode( 'woo_product_slider', [ $this, 'render_slider_shortcode' ] );
        
        // Simple Divi support - just make shortcode work in Divi
        add_filter( 'et_builder_module_shortcode_output', [ $this, 'divi_shortcode_output' ], 10, 3 );
        
        // Ensure WooCommerce styles are loaded
        add_action( 'wp_enqueue_scripts', [ $this, 'ensure_woocommerce_styles' ], 20 );
        
        // Setup admin menus under GrL'Ca - run later to ensure proper menu handling
        add_action( 'admin_menu', [ $this, 'setup_admin_menus' ], 20 );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_enqueue_scripts' ] );
    }
    
    /**
     * Setup admin menus under existing GrL'Ca dashboard
     */
    public function setup_admin_menus() {
        // Remove the default custom post type menu that gets auto-added
        remove_menu_page( 'edit.php?post_type=' . $this->post_type );
        
        // Add ONLY "Slider" as a submenu under the existing GrL'Ca menu
        // This leads to the slider dashboard where all controls are located
        add_submenu_page(
            'grlca-dashboard', // Parent is the existing GrL'Ca menu
            __( 'Slider Dashboard', 'grlca-woo-slider' ),
            __( 'Slider', 'grlca-woo-slider' ),
            'manage_options',
            'grlca-woo-slider',
            [ $this, 'display_grlca_slider_page' ]
        );
        
        // REMOVED: "Manage Product Sliders" direct link
        // REMOVED: "Customize Styles" direct link
        // These are now only accessible from the Slider Dashboard page
    }
    
    /**
     * Display GrL'Ca Slider main dashboard page
     * This is the main landing page when users click "Slider" in the GrL'Ca menu
     */
    public function display_grlca_slider_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions to access this page.', 'grlca-woo-slider' ) );
        }
        ?>
        <div class="wrap">
            <h1><?php _e( 'Slider Dashboard', 'grlca-woo-slider' ); ?></h1>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin: 20px 0;">
                <div style="background: #fff; padding: 20px; border-radius: 5px; border: 1px solid #e5e5e5;">
                    <h2><?php _e( 'Manage Product Sliders', 'grlca-woo-slider' ); ?></h2>
                    <p><?php _e( 'Create and manage your product sliders. Configure categories, product count, display options, and get shortcodes for each slider.', 'grlca-woo-slider' ); ?></p>
                    <a href="<?php echo admin_url( 'edit.php?post_type=' . $this->post_type ); ?>" class="button button-primary">
                        <?php _e( 'Manage Product Sliders', 'grlca-woo-slider' ); ?>
                    </a>
                </div>
                
                <div style="background: #fff; padding: 20px; border-radius: 5px; border: 1px solid #e5e5e5;">
                    <h2><?php _e( 'Customize Styles', 'grlca-woo-slider' ); ?></h2>
                    <p><?php _e( 'Customize the appearance of your sliders. Change colors, button styles, typography, and navigation settings.', 'grlca-woo-slider' ); ?></p>
                    <a href="<?php echo admin_url( 'admin.php?page=woo-slider-styling' ); ?>" class="button">
                        <?php _e( 'Customize Styles', 'grlca-woo-slider' ); ?>
                    </a>
                </div>
            </div>

            <div style="background: #fff; padding: 20px; border-radius: 5px; border: 1px solid #e5e5e5; margin-top: 20px;">
                <h2><?php _e( 'How to Use', 'grlca-woo-slider' ); ?></h2>
                <ol>
                    <li><strong><?php _e( 'Create Sliders:', 'grlca-woo-slider' ); ?></strong> <?php _e( 'Go to "Manage Product Sliders" to create new sliders and configure their settings', 'grlca-woo-slider' ); ?></li>
                    <li><strong><?php _e( 'Customize Appearance:', 'grlca-woo-slider' ); ?></strong> <?php _e( 'Use "Customize Styles" to change colors, buttons, and overall look', 'grlca-woo-slider' ); ?></li>
                    <li><strong><?php _e( 'Display Sliders:', 'grlca-woo-slider' ); ?></strong> <?php _e( 'Copy the shortcode from each slider and paste it in posts, pages, or widgets', 'grlca-woo-slider' ); ?></li>
                    <li><strong><?php _e( 'Shortcode Format:', 'grlca-woo-slider' ); ?></strong> <code>[woo_product_slider id="X"]</code></li>
                </ol>
            </div>
        </div>
        <?php
    }

    
    
    /** ----------------------------------
     *  ALL ORIGINAL FUNCTIONALITY
     * ---------------------------------- */
    
    public function register_slider_post_type() {
        $labels = [
            'name'               => __( 'Product Sliders', 'grlca-woo-slider' ),
            'singular_name'      => __( 'Product Slider', 'grlca-woo-slider' ),
            'menu_name'          => __( 'Product Sliders', 'grlca-woo-slider' ),
            'add_new'            => __( 'Add New', 'grlca-woo-slider' ),
            'add_new_item'       => __( 'Add New Slider', 'grlca-woo-slider' ),
            'edit_item'          => __( 'Edit Slider', 'grlca-woo-slider' ),
            'new_item'           => __( 'New Slider', 'grlca-woo-slider' ),
            'view_item'          => __( 'View Slider', 'grlca-woo-slider' ),
            'search_items'       => __( 'Search Sliders', 'grlca-woo-slider' ),
            'not_found'          => __( 'No sliders found', 'grlca-woo-slider' ),
            'not_found_in_trash' => __( 'No sliders found in Trash', 'grlca-woo-slider' ),
        ];

        $args = [
            'labels'             => $labels,
            'public'             => false,
            'show_ui'            => true,
            'show_in_menu'       => false, // Completely hidden from main menu
            'menu_icon'          => 'dashicons-images-alt2',
            'supports'           => [ 'title' ],
            'capability_type'    => 'post',
            'rewrite'            => false,
        ];

        register_post_type( $this->post_type, $args );
    }

    public function add_meta_boxes() {
        add_meta_box(
            'woo_slider_settings',
            __( 'Slider Settings', 'grlca-woo-slider' ),
            [ $this, 'render_slider_settings_box' ],
            $this->post_type,
            'normal',
            'high'
        );
    }

    public function render_slider_settings_box( $post ) {
        $meta = get_post_meta( $post->ID );
        $category       = $meta['category'][0] ?? '';
        $product_count  = $meta['product_count'][0] ?? 10;
        $show_cart      = ! empty( $meta['show_cart'][0] );
        $hide_out_of_stock = ! empty( $meta['hide_out_of_stock'][0] );
        $only_with_images = ! empty( $meta['only_with_images'][0] );
        $order_by       = $meta['order_by'][0] ?? 'date';
        $image_ratio    = $meta['image_ratio'][0] ?? '1-1';
        $slides_desktop = $meta['slides_desktop'][0] ?? 5;
        $slides_tablet  = $meta['slides_tablet'][0] ?? 3;
        $slides_mobile  = $meta['slides_mobile'][0] ?? 1;

        $categories = get_terms( [ 'taxonomy' => 'product_cat', 'hide_empty' => false ] );
        
        // Get the category name for display if a category is selected
        $selected_category_name = '';
        if ( $category ) {
            $selected_cat = get_term_by( 'slug', $category, 'product_cat' );
            if ( $selected_cat ) {
                $selected_category_name = $selected_cat->name;
            }
        }
        ?>
        
        <!-- Add Searchable Dropdown Styles -->
        <style>
        .searchable-dropdown {
            position: relative;
            width: 100%;
        }
        .searchable-dropdown input {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            margin-bottom: 5px;
        }
        .searchable-dropdown select {
            width: 100%;
            height: 150px;
            display: none;
            position: absolute;
            background: white;
            border: 1px solid #ddd;
            z-index: 1000;
            overflow-y: auto;
        }
        .searchable-dropdown select.show {
            display: block;
        }
        .searchable-dropdown .selected-value {
            background: #f0f0f1;
            padding: 8px;
            border-radius: 4px;
            margin-bottom: 5px;
            display: <?php echo $category ? 'block' : 'none'; ?>;
        }
        .searchable-dropdown .clear-selection {
            color: #dc3232;
            text-decoration: none;
            margin-left: 10px;
            font-size: 12px;
        }
        .searchable-dropdown .clear-selection:hover {
            color: #a00;
        }
        </style>

        <table class="form-table">
            <tr>
                <th><label><?php _e( 'Product Category', 'grlca-woo-slider' ); ?></label></th>
                <td>
                    <div class="searchable-dropdown">
                        <!-- Display selected category -->
                        <div class="selected-value" id="selected-category">
                            <strong><?php _e( 'Selected:', 'grlca-woo-slider' ); ?></strong> 
                            <span id="selected-category-name"><?php echo esc_html( $selected_category_name ); ?></span>
                            <a href="#" class="clear-selection" onclick="clearCategorySelection(event)"><?php _e( 'Clear', 'grlca-woo-slider' ); ?></a>
                            <input type="hidden" name="category" id="category-field" value="<?php echo esc_attr( $category ); ?>">
                        </div>
                        
                        <!-- Search input -->
                        <input type="text" id="category-search" placeholder="<?php _e( 'Search categories...', 'grlca-woo-slider' ); ?>" autocomplete="off">
                        
                        <!-- Dropdown list -->
                        <select id="category-select" multiple size="10">
                            <option value="">— <?php _e( 'All Categories', 'grlca-woo-slider' ); ?> —</option>
                            <?php foreach ( $categories as $cat ) : ?>
                                <option value="<?php echo esc_attr( $cat->slug ); ?>" data-name="<?php echo esc_attr( $cat->name ); ?>" <?php selected( $category, $cat->slug ); ?>>
                                    <?php echo esc_html( $cat->name ); ?> (<?php echo $cat->count; ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php _e( 'Start typing to search categories. Click on a category to select it.', 'grlca-woo-slider' ); ?></p>
                    </div>
                </td>
            </tr>

            <tr>
                <th><label><?php _e( 'Number of Products', 'grlca-woo-slider' ); ?></label></th>
                <td><input type="number" name="product_count" value="<?php echo esc_attr( $product_count ); ?>" min="1" max="50"></td>
            </tr>

            <tr>
                <th><label><?php _e( 'Product Order', 'grlca-woo-slider' ); ?></label></th>
                <td>
                    <select name="order_by">
                        <option value="date" <?php selected( $order_by, 'date' ); ?>><?php _e( 'Newest First', 'grlca-woo-slider' ); ?></option>
                        <option value="date-asc" <?php selected( $order_by, 'date-asc' ); ?>><?php _e( 'Oldest First', 'grlca-woo-slider' ); ?></option>
                        <option value="title" <?php selected( $order_by, 'title' ); ?>><?php _e( 'Title A-Z', 'grlca-woo-slider' ); ?></option>
                        <option value="title-desc" <?php selected( $order_by, 'title-desc' ); ?>><?php _e( 'Title Z-A', 'grlca-woo-slider' ); ?></option>
                        <option value="price" <?php selected( $order_by, 'price' ); ?>><?php _e( 'Price: Low to High', 'grlca-woo-slider' ); ?></option>
                        <option value="price-desc" <?php selected( $order_by, 'price-desc' ); ?>><?php _e( 'Price: High to Low', 'grlca-woo-slider' ); ?></option>
                        <option value="popularity" <?php selected( $order_by, 'popularity' ); ?>><?php _e( 'Best Selling', 'grlca-woo-slider' ); ?></option>
                        <option value="rating" <?php selected( $order_by, 'rating' ); ?>><?php _e( 'Top Rated', 'grlca-woo-slider' ); ?></option>
                        <option value="random" <?php selected( $order_by, 'random' ); ?>><?php _e( 'Random', 'grlca-woo-slider' ); ?></option>
                        <option value="featured" <?php selected( $order_by, 'featured' ); ?>><?php _e( 'Featured Products', 'grlca-woo-slider' ); ?></option>
                    </select>
                    <p class="description"><?php _e( 'Choose how products should be ordered in the slider.', 'grlca-woo-slider' ); ?></p>
                </td>
            </tr>

            <tr>
                <th><label><?php _e( 'Show Add to Cart', 'grlca-woo-slider' ); ?></label></th>
                <td>
                    <label><input type="checkbox" name="show_cart" value="1" <?php checked( $show_cart, true ); ?>> <?php _e( 'Enable', 'grlca-woo-slider' ); ?></label>
                </td>
            </tr>

            <tr>
                <th><label><?php _e( 'Hide Out of Stock Products', 'grlca-woo-slider' ); ?></label></th>
                <td>
                    <label><input type="checkbox" name="hide_out_of_stock" value="1" <?php checked( $hide_out_of_stock, true ); ?>> <?php _e( 'Enable', 'grlca-woo-slider' ); ?></label>
                    <p class="description"><?php _e( 'When enabled, products that are out of stock will not be displayed in the slider.', 'grlca-woo-slider' ); ?></p>
                </td>
            </tr>

            <tr>
                <th><label><?php _e( 'Only Products with Custom Images', 'grlca-woo-slider' ); ?></label></th>
                <td>
                    <label><input type="checkbox" name="only_with_images" value="1" <?php checked( $only_with_images, true ); ?>> <?php _e( 'Enable', 'grlca-woo-slider' ); ?></label>
                    <p class="description"><?php _e( 'When enabled, only products with custom uploaded images will be shown. Products using WooCommerce placeholder images will be excluded.', 'grlca-woo-slider' ); ?></p>
                </td>
            </tr>

            <tr>
                <th><label><?php _e( 'Image Aspect Ratio', 'grlca-woo-slider' ); ?></label></th>
                <td>
                    <select name="image_ratio">
                        <option value="1-1" <?php selected( $image_ratio, '1-1' ); ?>><?php _e( 'Square (1:1)', 'grlca-woo-slider' ); ?></option>
                        <option value="4-3" <?php selected( $image_ratio, '4-3' ); ?>><?php _e( 'Standard (4:3)', 'grlca-woo-slider' ); ?></option>
                    </select>
                    <p class="description"><?php _e( 'Choose the aspect ratio for product images. All images will be cropped to maintain consistent appearance.', 'grlca-woo-slider' ); ?></p>
                </td>
            </tr>

            <tr><th colspan="2"><h4><?php _e( 'Responsive Slides per View', 'grlca-woo-slider' ); ?></h4></th></tr>

            <tr>
                <th><label><?php _e( 'Desktop (≥1024px)', 'grlca-woo-slider' ); ?></label></th>
                <td><input type="number" name="slides_desktop" value="<?php echo esc_attr( $slides_desktop ); ?>" min="1" max="10"></td>
            </tr>
            <tr>
                <th><label><?php _e( 'Tablet (≥768px)', 'grlca-woo-slider' ); ?></label></th>
                <td><input type="number" name="slides_tablet" value="<?php echo esc_attr( $slides_tablet ); ?>" min="1" max="10"></td>
            </tr>
            <tr>
                <th><label><?php _e( 'Mobile (&lt;768px)', 'grlca-woo-slider' ); ?></label></th>
                <td><input type="number" name="slides_mobile" value="<?php echo esc_attr( $slides_mobile ); ?>" min="1" max="10"></td>
            </tr>
        </table>

        <p><strong><?php _e( 'Shortcode:', 'grlca-woo-slider' ); ?></strong> <code>[woo_product_slider id="<?php echo esc_attr( $post->ID ); ?>"]</code></p>
        
        <div style="background: #f0f0f1; padding: 15px; margin-top: 20px; border-left: 4px solid #007cba;">
            <h4 style="margin-top: 0;"><?php _e( 'Custom Styling', 'grlca-woo-slider' ); ?></h4>
            <p style="margin-bottom: 0;">
                <?php 
                printf(
                    __( 'You can customize the appearance of your sliders from the <a href="%s">Customize Styles</a> page.', 'grlca-woo-slider' ),
                    admin_url( 'admin.php?page=woo-slider-styling' )
                ); 
                ?>
            </p>
        </div>

        <!-- Searchable Dropdown JavaScript -->
        <script>
        jQuery(document).ready(function($) {
            const $searchInput = $('#category-search');
            const $select = $('#category-select');
            const $selectedDisplay = $('#selected-category');
            const $categoryField = $('#category-field');
            const $selectedName = $('#selected-category-name');

            // Show/hide dropdown on focus
            $searchInput.on('focus', function() {
                $select.addClass('show');
                filterCategories();
            });

            // Hide dropdown when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.searchable-dropdown').length) {
                    $select.removeClass('show');
                }
            });

            // Filter categories based on search input
            $searchInput.on('input', function() {
                filterCategories();
            });

            // Handle category selection
            $select.on('click', 'option', function() {
                const value = $(this).val();
                const name = $(this).data('name');
                
                $categoryField.val(value);
                $selectedName.text(name || '<?php _e( "All Categories", "grlca-woo-slider" ); ?>');
                $selectedDisplay.show();
                $select.removeClass('show');
                $searchInput.val('');
                
                // Update all options to remove selection
                $select.find('option').prop('selected', false);
                // Select the clicked option
                $(this).prop('selected', true);
                
                filterCategories(); // Re-filter to show all options
            });

            function filterCategories() {
                const searchTerm = $searchInput.val().toLowerCase();
                let hasVisibleOptions = false;

                $select.find('option').each(function() {
                    const $option = $(this);
                    const text = $option.text().toLowerCase();
                    
                    if (text.includes(searchTerm)) {
                        $option.show();
                        hasVisibleOptions = true;
                    } else {
                        $option.hide();
                    }
                });

                // If no results, show a message
                if (!hasVisibleOptions) {
                    $select.append('<option value="" disabled><?php _e( "No categories found", "grlca-woo-slider" ); ?></option>');
                }
            }

            // Initialize with current selection
            if ($categoryField.val()) {
                $select.find('option[value="' + $categoryField.val() + '"]').prop('selected', true);
            }
        });

        function clearCategorySelection(event) {
            event.preventDefault();
            const $categoryField = $('#category-field');
            const $selectedDisplay = $('#selected-category');
            const $select = $('#category-select');
            
            $categoryField.val('');
            $selectedDisplay.hide();
            $select.find('option').prop('selected', false);
            $select.find('option[value=""]').prop('selected', true);
        }
        </script>
        <?php
    }

    public function save_slider_meta( $post_id ) {
        if ( get_post_type( $post_id ) !== $this->post_type ) return;
        
        // Verify nonce for security
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'update-post_' . $post_id ) ) {
            return;
        }

        $fields = [ 'category', 'product_count', 'show_cart', 'hide_out_of_stock', 'only_with_images', 'order_by', 'image_ratio', 'slides_desktop', 'slides_tablet', 'slides_mobile' ];

        foreach ( $fields as $field ) {
            if ( isset( $_POST[$field] ) ) {
                update_post_meta( $post_id, $field, sanitize_text_field( $_POST[$field] ) );
            } else {
                delete_post_meta( $post_id, $field );
            }
        }
    }

    public function admin_enqueue_scripts( $hook ) {
        // Check if hook is valid before using strpos
        if ( empty( $hook ) ) {
            return;
        }
        
        if ( strpos( $hook, 'woo-slider-styling' ) !== false ) {
            wp_enqueue_style( 'wp-color-picker' );
            wp_enqueue_script( 'wp-color-picker' );
        }
    }

    public function register_settings() {
        // Button Styles
        register_setting( 'woo_slider_styling', 'woo_slider_button_bg_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_text_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_hover_bg_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_hover_text_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_padding' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_border_radius' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_font_size' );
        register_setting( 'woo_slider_styling', 'woo_slider_button_font_weight' );
        
        // Product Card Styles
        register_setting( 'woo_slider_styling', 'woo_slider_card_bg_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_card_border_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_card_border_radius' );
        register_setting( 'woo_slider_styling', 'woo_slider_card_padding' );
        
        // Text Styles
        register_setting( 'woo_slider_styling', 'woo_slider_title_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_title_font_size' );
        register_setting( 'woo_slider_styling', 'woo_slider_price_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_price_font_size' );
        
        // Navigation Styles
        register_setting( 'woo_slider_styling', 'woo_slider_nav_bg_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_nav_text_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_nav_hover_bg_color' );
        register_setting( 'woo_slider_styling', 'woo_slider_nav_hover_text_color' );
    }

    public function styling_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e( 'Customize Slider Styles', 'grlca-woo-slider' ); ?></h1>
            
            <form method="post" action="options.php">
                <?php settings_fields( 'woo_slider_styling' ); ?>
                
                <div class="woo-slider-styling-tabs">
                    <h2 class="nav-tab-wrapper">
                        <a href="#tab-button" class="nav-tab nav-tab-active"><?php _e( 'Button Styles', 'grlca-woo-slider' ); ?></a>
                        <a href="#tab-card" class="nav-tab"><?php _e( 'Product Card', 'grlca-woo-slider' ); ?></a>
                        <a href="#tab-text" class="nav-tab"><?php _e( 'Text Styles', 'grlca-woo-slider' ); ?></a>
                        <a href="#tab-navigation" class="nav-tab"><?php _e( 'Navigation', 'grlca-woo-slider' ); ?></a>
                    </h2>

                    <div id="tab-button" class="tab-content active">
                        <table class="form-table">
                            <tr>
                                <th><label><?php _e( 'Button Background Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_button_bg_color" value="<?php echo esc_attr( get_option( 'woo_slider_button_bg_color', '#007cba' ) ); ?>" class="color-picker" data-default-color="#007cba">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Text Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_button_text_color" value="<?php echo esc_attr( get_option( 'woo_slider_button_text_color', '#ffffff' ) ); ?>" class="color-picker" data-default-color="#ffffff">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Hover Background', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_button_hover_bg_color" value="<?php echo esc_attr( get_option( 'woo_slider_button_hover_bg_color', '#005a87' ) ); ?>" class="color-picker" data-default-color="#005a87">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Hover Text Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_button_hover_text_color" value="<?php echo esc_attr( get_option( 'woo_slider_button_hover_text_color', '#ffffff' ) ); ?>" class="color-picker" data-default-color="#ffffff">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Padding', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_button_padding" value="<?php echo esc_attr( get_option( 'woo_slider_button_padding', '10px 20px' ) ); ?>" placeholder="10px 20px">
                                    <p class="description"><?php _e( 'Format: top/bottom left/right (e.g., 10px 20px)', 'grlca-woo-slider' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Border Radius', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="number" name="woo_slider_button_border_radius" value="<?php echo esc_attr( get_option( 'woo_slider_button_border_radius', '3' ) ); ?>" min="0" max="50"> px
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Font Size', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="number" name="woo_slider_button_font_size" value="<?php echo esc_attr( get_option( 'woo_slider_button_font_size', '14' ) ); ?>" min="10" max="24"> px
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Button Font Weight', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <select name="woo_slider_button_font_weight">
                                        <option value="normal" <?php selected( get_option( 'woo_slider_button_font_weight', 'bold' ), 'normal' ); ?>><?php _e( 'Normal', 'grlca-woo-slider' ); ?></option>
                                        <option value="bold" <?php selected( get_option( 'woo_slider_button_font_weight', 'bold' ), 'bold' ); ?>><?php _e( 'Bold', 'grlca-woo-slider' ); ?></option>
                                        <option value="600" <?php selected( get_option( 'woo_slider_button_font_weight', 'bold' ), '600' ); ?>>600</option>
                                        <option value="700" <?php selected( get_option( 'woo_slider_button_font_weight', 'bold' ), '700' ); ?>>700</option>
                                    </select>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div id="tab-card" class="tab-content">
                        <table class="form-table">
                            <tr>
                                <th><label><?php _e( 'Card Background Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_card_bg_color" value="<?php echo esc_attr( get_option( 'woo_slider_card_bg_color', '#ffffff' ) ); ?>" class="color-picker" data-default-color="#ffffff">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Card Border Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_card_border_color" value="<?php echo esc_attr( get_option( 'woo_slider_card_border_color', '#e5e5e5' ) ); ?>" class="color-picker" data-default-color="#e5e5e5">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Card Border Radius', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="number" name="woo_slider_card_border_radius" value="<?php echo esc_attr( get_option( 'woo_slider_card_border_radius', '5' ) ); ?>" min="0" max="20"> px
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Card Padding', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="number" name="woo_slider_card_padding" value="<?php echo esc_attr( get_option( 'woo_slider_card_padding', '15' ) ); ?>" min="5" max="30"> px
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div id="tab-text" class="tab-content">
                        <table class="form-table">
                            <tr>
                                <th><label><?php _e( 'Product Title Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_title_color" value="<?php echo esc_attr( get_option( 'woo_slider_title_color', '#333333' ) ); ?>" class="color-picker" data-default-color="#333333">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Product Title Font Size', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="number" name="woo_slider_title_font_size" value="<?php echo esc_attr( get_option( 'woo_slider_title_font_size', '14' ) ); ?>" min="10" max="24"> px
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Price Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_price_color" value="<?php echo esc_attr( get_option( 'woo_slider_price_color', '#333333' ) ); ?>" class="color-picker" data-default-color="#333333">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Price Font Size', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="number" name="woo_slider_price_font_size" value="<?php echo esc_attr( get_option( 'woo_slider_price_font_size', '16' ) ); ?>" min="10" max="24"> px
                                </td>
                            </tr>
                        </table>
                    </div>

                    <div id="tab-navigation" class="tab-content">
                        <table class="form-table">
                            <tr>
                                <th><label><?php _e( 'Navigation Background Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_nav_bg_color" value="<?php echo esc_attr( get_option( 'woo_slider_nav_bg_color', 'rgba(255,255,255,0.8)' ) ); ?>" class="color-picker" data-default-color="rgba(255,255,255,0.8)">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Navigation Text Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_nav_text_color" value="<?php echo esc_attr( get_option( 'woo_slider_nav_text_color', '#333333' ) ); ?>" class="color-picker" data-default-color="#333333">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Navigation Hover Background', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_nav_hover_bg_color" value="<?php echo esc_attr( get_option( 'woo_slider_nav_hover_bg_color', 'rgba(255,255,255,0.9)' ) ); ?>" class="color-picker" data-default-color="rgba(255,255,255,0.9)">
                                </td>
                            </tr>
                            <tr>
                                <th><label><?php _e( 'Navigation Hover Text Color', 'grlca-woo-slider' ); ?></label></th>
                                <td>
                                    <input type="text" name="woo_slider_nav_hover_text_color" value="<?php echo esc_attr( get_option( 'woo_slider_nav_hover_text_color', '#000000' ) ); ?>" class="color-picker" data-default-color="#000000">
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
                
                <?php submit_button( __( 'Save Styles', 'grlca-woo-slider' ) ); ?>
            </form>

            <div style="margin-top: 30px; padding: 20px; background: #f9f9f9; border-left: 4px solid #007cba;">
                <h3><?php _e( 'Preview', 'grlca-woo-slider' ); ?></h3>
                <p><?php _e( 'Your changes will be applied to all product sliders immediately after saving.', 'grlca-woo-slider' ); ?></p>
            </div>
        </div>

        <style>
        .woo-slider-styling-tabs .tab-content {
            display: none;
        }
        .woo-slider-styling-tabs .tab-content.active {
            display: block;
        }
        .nav-tab-wrapper {
            margin-bottom: 20px;
        }
        .color-picker {
            width: 100px;
        }
        </style>

        <script>
        jQuery(document).ready(function($) {
            // Tab switching
            $('.nav-tab').on('click', function(e) {
                e.preventDefault();
                $('.nav-tab').removeClass('nav-tab-active');
                $('.tab-content').removeClass('active');
                
                $(this).addClass('nav-tab-active');
                $($(this).attr('href')).addClass('active');
            });

            // Initialize color pickers
            $('.color-picker').wpColorPicker();
        });
        </script>
        <?php
    }

    /** ----------------------------------
     *  Get WooCommerce Placeholder Image ID
     * ---------------------------------- */
    private function get_placeholder_image_id() {
        // Get the placeholder image ID from WooCommerce settings
        $placeholder_id = get_option( 'woocommerce_placeholder_image', 0 );
        
        // If no custom placeholder is set, try to find the default one
        if ( ! $placeholder_id ) {
            // Look for the default WooCommerce placeholder image
            $default_placeholder = get_posts( [
                'post_type' => 'attachment',
                'posts_per_page' => 1,
                's' => 'woocommerce-placeholder',
                'post_status' => 'inherit'
            ] );
            
            if ( ! empty( $default_placeholder ) ) {
                $placeholder_id = $default_placeholder[0]->ID;
            }
        }
        
        return absint( $placeholder_id );
    }

    /** ----------------------------------
     *  Check if product has custom image
     * ---------------------------------- */
    private function has_custom_image( $product ) {
        $image_id = $product->get_image_id();
        
        // If no image ID, definitely no custom image
        if ( ! $image_id ) {
            return false;
        }
        
        // Get the placeholder image ID
        $placeholder_id = $this->get_placeholder_image_id();
        
        // If product image ID matches placeholder ID, it's not a custom image
        if ( $image_id == $placeholder_id ) {
            return false;
        }
        
        return true;
    }

    /** ----------------------------------
     *  Generate Dynamic CSS from Settings
     * ---------------------------------- */
    private function generate_dynamic_css() {
        $css = '';
        
        // Add loading state to prevent FOUC (Flash of Unstyled Content)
        $css .= ".woo-product-slider-container { opacity: 0; transition: opacity 0.3s ease; }";
        $css .= ".woo-product-slider-container.swiper-loaded { opacity: 1; }";
        
        // Button Styles
        $css .= ".woo-product-slider .add_to_cart_button, .woo-product-slider .button {";
        $css .= "background-color: " . get_option( 'woo_slider_button_bg_color', '#007cba' ) . " !important;";
        $css .= "color: " . get_option( 'woo_slider_button_text_color', '#ffffff' ) . " !important;";
        $css .= "padding: " . get_option( 'woo_slider_button_padding', '10px 20px' ) . " !important;";
        $css .= "border-radius: " . get_option( 'woo_slider_button_border_radius', '3' ) . "px !important;";
        $css .= "font-size: " . get_option( 'woo_slider_button_font_size', '14' ) . "px !important;";
        $css .= "font-weight: " . get_option( 'woo_slider_button_font_weight', 'bold' ) . " !important;";
        $css .= "border: none !important;";
        $css .= "min-height: 42px !important;";
        $css .= "display: inline-flex !important;";
        $css .= "align-items: center !important;";
        $css .= "justify-content: center !important;";
        $css .= "}";
        
        $css .= ".woo-product-slider .add_to_cart_button:hover, .woo-product-slider .button:hover {";
        $css .= "background-color: " . get_option( 'woo_slider_button_hover_bg_color', '#005a87' ) . " !important;";
        $css .= "color: " . get_option( 'woo_slider_button_hover_text_color', '#ffffff' ) . " !important;";
        $css .= "}";
        
        // Product Card Styles
        $css .= ".woo-product-slider .product {";
        $css .= "background-color: " . get_option( 'woo_slider_card_bg_color', '#ffffff' ) . " !important;";
        $css .= "border: 1px solid " . get_option( 'woo_slider_card_border_color', '#e5e5e5' ) . " !important;";
        $css .= "border-radius: " . get_option( 'woo_slider_card_border_radius', '5' ) . "px !important;";
        $css .= "padding: " . get_option( 'woo_slider_card_padding', '15' ) . "px !important;";
        $css .= "}";
        
        // Text Styles
        $css .= ".woo-product-slider .woocommerce-loop-product__title {";
        $css .= "color: " . get_option( 'woo_slider_title_color', '#333333' ) . " !important;";
        $css .= "font-size: " . get_option( 'woo_slider_title_font_size', '14' ) . "px !important;";
        $css .= "}";
        
        $css .= ".woo-product-slider .price {";
        $css .= "color: " . get_option( 'woo_slider_price_color', '#333333' ) . " !important;";
        $css .= "font-size: " . get_option( 'woo_slider_price_font_size', '16' ) . "px !important;";
        $css .= "}";
        
        // Navigation Styles
        $css .= ".woo-product-slider .swiper-button-prev, .woo-product-slider .swiper-button-next {";
        $css .= "background: " . get_option( 'woo_slider_nav_bg_color', 'rgba(255,255,255,0.8)' ) . " !important;";
        $css .= "color: " . get_option( 'woo_slider_nav_text_color', '#333333' ) . " !important;";
        $css .= "}";
        
        $css .= ".woo-product-slider .swiper-button-prev:hover, .woo-product-slider .swiper-button-next:hover {";
        $css .= "background: " . get_option( 'woo_slider_nav_hover_bg_color', 'rgba(255,255,255,0.9)' ) . " !important;";
        $css .= "color: " . get_option( 'woo_slider_nav_hover_text_color', '#000000' ) . " !important;";
        $css .= "}";
        
        return $css;
    }

    /** ----------------------------------
     *  Enqueue Scripts
     * ---------------------------------- */
    public function enqueue_scripts() {
        wp_enqueue_style( 'swiper-css', 'https://unpkg.com/swiper@9/swiper-bundle.min.css' );
        wp_enqueue_script( 'swiper-js', 'https://unpkg.com/swiper@9/swiper-bundle.min.js', [ 'jquery' ], null, true );

        // Base CSS
        $base_css = "
            .woo-product-slider-container { 
                margin: 40px 0; 
            }
            .woo-product-slider .swiper-slide {
                background: none;
                border: none;
                padding: 10px;
                height: auto;
            }
            .woo-slider-product {
                height: 100%;
            }
            .woo-product-slider .product {
                text-align: center;
                height: 100%;
                display: flex;
                flex-direction: column;
            }
            .woo-product-slider .woocommerce-loop-product__link {
                display: block;
                margin-bottom: 10px;
            }
            .woo-product-slider .woocommerce-loop-product__title {
                line-height: 1.4;
                margin: 10px 0 5px 0;
                min-height: 40px;
            }
            .woo-product-slider .price {
                display: block;
                margin: 5px 0 10px 0;
                font-weight: bold;
            }
            .woo-product-slider .star-rating {
                margin: 0 auto 5px auto;
                font-size: 12px;
            }
            .woo-product-slider .out-of-stock {
                color: #ff4b4b;
                font-weight: bold;
                margin-top: 10px;
                display: block;
            }
            .woo-product-slider .add_to_cart_button,
            .woo-product-slider .product_type_simple,
            .woo-product-slider .product_type_variable,
            .woo-product-slider .single_add_to_cart_button {
                display: inline-block !important;
                width: auto !important;
            }
            
            /* NEW: Align Add to Cart buttons at the bottom */
            .woo-product-slider .woocommerce-product-actions {
                margin-top: auto;
                padding-top: 15px;
                min-height: 50px;
                display: flex;
                align-items: flex-end;
                justify-content: center;
            }
            
            /* NEW: Ensure consistent button height */
            .woo-product-slider .woocommerce-product-actions .button,
            .woo-product-slider .woocommerce-product-actions .add_to_cart_button,
            .woo-product-slider .woocommerce-product-actions .out-of-stock {
                min-height: 42px;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                white-space: nowrap;
            }
            
            /* NEW: Style for out of stock text to match button height */
            .woo-product-slider .woocommerce-product-actions .out-of-stock {
                padding: 10px 20px;
                border: 1px solid #e5e5e5;
                border-radius: 3px;
                background: #f8f8f8;
                color: #666;
            }

            /* NEW: Aspect ratio image styling */
            .woo-product-slider .woocommerce-loop-product__link {
                position: relative;
                display: block;
                overflow: hidden;
            }
            
            .woo-product-slider .woocommerce-loop-product__link img {
                width: 100%;
                height: auto;
                display: block;
                transition: transform 0.3s ease;
            }
            
            .woo-product-slider .woocommerce-loop-product__link:hover img {
                transform: scale(1.05);
            }

            .swiper-button-prev, .swiper-button-next { 
                border-radius: 50%;
                width: 40px;
                height: 40px;
            }
            .swiper-button-prev:after, .swiper-button-next:after {
                font-size: 18px;
            }
        ";

        // Generate dynamic CSS from settings
        $dynamic_css = $this->generate_dynamic_css();
        
        // Combine base CSS with dynamic CSS
        wp_add_inline_style( 'swiper-css', $base_css . $dynamic_css );
    }

    /** ----------------------------------
     *  Ensure WooCommerce Styles
     * ---------------------------------- */
    public function ensure_woocommerce_styles() {
        // Make sure WooCommerce frontend styles are loaded
        if ( function_exists( 'WC' ) ) {
            wp_enqueue_style( 'woocommerce-general' );
            wp_enqueue_style( 'woocommerce-layout' );
            wp_enqueue_style( 'woocommerce-smallscreen' );
        }
    }

    /** ----------------------------------
     *  Shortcode Renderer with Image Filtering
     * ---------------------------------- */
    public function render_slider_shortcode( $atts ) {
        $atts = shortcode_atts( [ 'id' => '' ], $atts, 'woo_product_slider' );
        $id = absint( $atts['id'] );
        if ( ! $id ) return '<p>No slider ID provided.</p>';

        $meta = get_post_meta( $id );
        $category       = $meta['category'][0] ?? '';
        $product_count  = $meta['product_count'][0] ?? 10;
        $show_cart      = ! empty( $meta['show_cart'][0] );
        $hide_out_of_stock = ! empty( $meta['hide_out_of_stock'][0] );
        $only_with_images = ! empty( $meta['only_with_images'][0] );
        $order_by       = $meta['order_by'][0] ?? 'date';
        $image_ratio    = $meta['image_ratio'][0] ?? '1-1';
        $slides_desktop = $meta['slides_desktop'][0] ?? 5;
        $slides_tablet  = $meta['slides_tablet'][0] ?? 3;
        $slides_mobile  = $meta['slides_mobile'][0] ?? 1;

        $args = [
            'post_type'      => 'product',
            'posts_per_page' => $only_with_images ? 100 : $product_count, // Get more products if we need to filter by images
            'post_status'    => 'publish',
            'meta_query'     => [],
            'tax_query'      => [],
        ];

        // Handle different ordering options
        switch( $order_by ) {
            case 'date':
                $args['orderby'] = 'date';
                $args['order'] = 'DESC';
                break;
            case 'date-asc':
                $args['orderby'] = 'date';
                $args['order'] = 'ASC';
                break;
            case 'title':
                $args['orderby'] = 'title';
                $args['order'] = 'ASC';
                break;
            case 'title-desc':
                $args['orderby'] = 'title';
                $args['order'] = 'DESC';
                break;
            case 'price':
                $args['orderby'] = 'meta_value_num';
                $args['meta_key'] = '_price';
                $args['order'] = 'ASC';
                break;
            case 'price-desc':
                $args['orderby'] = 'meta_value_num';
                $args['meta_key'] = '_price';
                $args['order'] = 'DESC';
                break;
            case 'popularity':
                $args['orderby'] = 'meta_value_num';
                $args['meta_key'] = 'total_sales';
                $args['order'] = 'DESC';
                break;
            case 'rating':
                $args['orderby'] = 'meta_value_num';
                $args['meta_key'] = '_wc_average_rating';
                $args['order'] = 'DESC';
                break;
            case 'random':
                $args['orderby'] = 'rand';
                break;
            case 'featured':
                $args['tax_query'][] = [
                    'taxonomy' => 'product_visibility',
                    'field'    => 'name',
                    'terms'    => 'featured',
                ];
                $args['orderby'] = 'date';
                $args['order'] = 'DESC';
                break;
            default:
                $args['orderby'] = 'date';
                $args['order'] = 'DESC';
        }

        // Add category filter if selected
        if ( ! empty( $category ) ) {
            $args['tax_query'][] = [
                'taxonomy' => 'product_cat',
                'field'    => 'slug',
                'terms'    => $category,
            ];
        }

        // Add stock status filter if enabled
        if ( $hide_out_of_stock ) {
            $args['meta_query'][] = [
                'key'     => '_stock_status',
                'value'   => 'outofstock',
                'compare' => '!='
            ];
        }

        // Ensure tax query works properly
        if ( count( $args['tax_query'] ) > 1 ) {
            $args['tax_query']['relation'] = 'AND';
        }

        $loop = new WP_Query( $args );
        
        $filtered_products = [];
        $products_with_images = 0;
        
        if ( $loop->have_posts() ) {
            while ( $loop->have_posts() ) {
                $loop->the_post();
                global $product;
                
                // Skip out of stock products if enabled
                if ( $hide_out_of_stock && ! $product->is_in_stock() ) {
                    continue;
                }
                
                // Skip products without custom images if enabled
                if ( $only_with_images && ! $this->has_custom_image( $product ) ) {
                    continue;
                }
                
                // Add product to filtered list
                $filtered_products[] = $product;
                $products_with_images++;
                
                // Stop if we have enough products
                if ( $products_with_images >= $product_count ) {
                    break;
                }
            }
        }
        
        wp_reset_postdata();
        
        // Check if we have any products after filtering
        if ( empty( $filtered_products ) ) {
            if ( $only_with_images ) {
                return '<p>No products with custom images found for the selected criteria.</p>';
            } else {
                return '<p>No products found.</p>';
            }
        }

        $uid = 'slider_' . $id . '_' . uniqid();

        ob_start(); ?>
        <div class="woo-product-slider-container">
            <div class="swiper woo-product-slider <?php echo esc_attr( $uid ); ?>">
                <div class="swiper-wrapper">
                    <?php foreach ( $filtered_products as $product ) : ?>
                        <div class="swiper-slide">
                            <div <?php wc_product_class( '', $product ); ?>>
                                <a href="<?php echo esc_url( get_permalink( $product->get_id() ) ); ?>" class="woocommerce-loop-product__link">
                                    <?php echo $product->get_image( 'woocommerce_thumbnail' ); ?>
                                </a>
                                
                                <h2 class="woocommerce-loop-product__title">
                                    <?php echo $product->get_name(); ?>
                                </h2>
                                
                                <?php if ( $product->get_rating_count() ) : ?>
                                    <div class="woocommerce-product-rating">
                                        <?php echo wc_get_rating_html( $product->get_average_rating() ); ?>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="price">
                                    <?php echo $product->get_price_html(); ?>
                                    <?php if ( ! $product->is_in_stock() && ! $hide_out_of_stock ) : ?>
                                        <span class="out-of-stock"><?php _e( '(Out of Stock)', 'grlca-woo-slider' ); ?></span>
                                    <?php endif; ?>
                                </div>
                                
                                <?php if ( $show_cart ) : ?>
                                    <div class="woocommerce-product-actions">
                                        <?php
                                        if ( $product->is_in_stock() ) {
                                            echo apply_filters( 'woocommerce_loop_add_to_cart_link',
                                                sprintf( '<a href="%s" data-quantity="1" class="%s" %s>%s</a>',
                                                    esc_url( $product->add_to_cart_url() ),
                                                    'button',
                                                    '',
                                                    esc_html( $product->add_to_cart_text() )
                                                ),
                                            $product );
                                        } else {
                                            echo '<span class="out-of-stock">' . __( 'Out of Stock', 'grlca-woo-slider' ) . '</span>';
                                        }
                                        ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="swiper-button-prev"></div>
                <div class="swiper-button-next"></div>
            </div>
        </div>

        <style>
        /* Aspect ratio CSS based on selection */
        .woo-product-slider .woocommerce-loop-product__link {
            position: relative;
            overflow: hidden;
        }
        
        <?php if ( $image_ratio === '1-1' ) : ?>
            /* Square (1:1) ratio */
            .woo-product-slider .woocommerce-loop-product__link {
                aspect-ratio: 1 / 1;
            }
            .woo-product-slider .woocommerce-loop-product__link img {
                width: 100%;
                height: 100%;
                object-fit: cover;
                object-position: center;
            }
        <?php elseif ( $image_ratio === '4-3' ) : ?>
            /* Standard (4:3) ratio */
            .woo-product-slider .woocommerce-loop-product__link {
                aspect-ratio: 4 / 3;
            }
            .woo-product-slider .woocommerce-loop-product__link img {
                width: 100%;
                height: 100%;
                object-fit: cover;
                object-position: center;
            }
        <?php endif; ?>
        </style>

        <script>
        jQuery(document).ready(function($){
            if (typeof Swiper !== 'undefined') {
                var slider = new Swiper('.<?php echo esc_js( $uid ); ?>', {
                    slidesPerView: <?php echo esc_js( $slides_mobile ); ?>,
                    spaceBetween: 10,
                    navigation: {
                        nextEl: '.swiper-button-next',
                        prevEl: '.swiper-button-next',
                    },
                    breakpoints: {
                        768: {
                            slidesPerView: <?php echo esc_js( $slides_tablet ); ?>,
                            spaceBetween: 15,
                        },
                        1024: {
                            slidesPerView: <?php echo esc_js( $slides_desktop ); ?>,
                            spaceBetween: 20,
                        },
                    }
                });

                // Mark slider as loaded to prevent FOUC
                $('.<?php echo esc_js( $uid ); ?>').closest('.woo-product-slider-container').addClass('swiper-loaded');
            }
        });
        </script>
        <?php
        return ob_get_clean();
    }

    /** ----------------------------------
     *  Divi Shortcode Output Filter
     * ---------------------------------- */
    public function divi_shortcode_output( $output, $render_slug, $module ) {
        // Ensure our shortcodes work in Divi Builder
        if ( false !== strpos( $output, '[woo_product_slider' ) ) {
            $output = do_shortcode( $output );
        }
        return $output;
    }
}

// Initialize the plugin
new GrLca_Woo_Product_Slider_Manager();